import * as constant from './constants.js'
import * as util from './util.js'
import * as bolt from './bolt.js'
import * as api from './api.js'
import axios from 'axios'
import fs from 'fs'
import tempWrite from 'temp-write'

function extractSuspiciousIpAddressFromMessageTags (message) {
  const tagsAttachment = message.attachments[0].fields.filter(x => x.title === 'Tags')
  const tags = {}
  let suspiciousIpAddress
  for (const key in tagsAttachment) {
    // split csv tags attachment value into array for each tag
    const split = tagsAttachment[key].value.split(',').map(e => e.split(':'))
    split.forEach((element) => {
      tags[element[0].trim()] = element[1].trim()
    })

    if (tags['context.query_params.ip']) {
      suspiciousIpAddress = tags['context.query_params.ip']
      return suspiciousIpAddress
    }
  }
  return suspiciousIpAddress
}

function generateModal (suspiciousIpAddress, body) {
  const privateMetadata = getPrivateMetadata(body)
  return {
    type: 'modal',
    callback_id: 'spam_slayer_modal',
    title: {
      type: 'plain_text',
      text: 'Spam Slayer'
    },
    blocks: [
      {
        type: 'section',
        text: {
          type: 'mrkdwn',
          text: `*Suspicious IP Address:* ${suspiciousIpAddress}\n\n` +
                            ' *Submitting this form will:* \n' +
                            `  - Add ${suspiciousIpAddress} to the traffic IP blocklist\n` +
                            '  - Demonetize clicks after the selected date\n\n'
        }
      },
      {
        type: 'divider'
      },
      {
        block_id: 'checkbox-block',
        type: 'section',
        text: {
          type: 'plain_text',
          // blank space string to bypass needing a title
          text: ' '
        },
        accessory: {
          type: 'checkboxes',
          action_id: 'checkbox-action',
          options: [
            {
              value: 'true',
              text: {
                type: 'plain_text',
                text: 'Block This IP Address'
              }
            }
          ]
        }
      },
      {
        block_id: 'datepicker-block',
        type: 'input',
        element: {
          type: 'datepicker',
          initial_date: util.getCurrentMonthStartDate(),
          placeholder: {
            type: 'plain_text',
            text: 'Select a date'
          },
          action_id: 'datepicker-action'
        },
        label: {
          type: 'plain_text',
          text: 'Demonetize Clicks After:'
        }
      },
      {
        type: 'section',
        text: {
          type: 'mrkdwn',
          text: '* *Note:* The demonetization date should _usually_ be the start of the month'
        }
      }
    ],
    submit: {
      type: 'plain_text',
      text: 'Submit'
    },
    close: {
      type: 'plain_text',
      text: 'Cancel'
    },
    private_metadata: `${privateMetadata}`
  }
}

function getPrivateMetadata (body) {
  const ipAddress = body.actions[0].value
  const user = body.user.id
  const userEmail = body.user.email
  const responseUrl = body.response_url
  const thread = body.message.ts

  return `{"suspiciousIpAddress": "${ipAddress}","user": "${user}","userEmail": "${userEmail}","responseUrl": "${responseUrl}","thread": "${thread}"}`
}

export function handleSpamSlayerAlertMessage (message, say) {
  bolt.verifyChannelPermissions(message.channel, say)
  const suspiciousIpAddress = extractSuspiciousIpAddressFromMessageTags(message)
  if (typeof suspiciousIpAddress !== 'undefined') {
    sendSupiciousIpAlertSlackResponse(suspiciousIpAddress, say)
  }
}

export async function isSpamSlayerAlertMessageMiddleware ({ message, next }) {
  const isSpamSlayerAlertMessage = message?.bot_profile?.name === constant.dataDogAppName &&
        message?.attachments[0]?.title.includes('ICP Click IP &amp; User Agent') &&
        !message?.attachments[0]?.title.includes('Warn') &&
        !message?.attachments[0]?.title.includes('Recovered')
  if (isSpamSlayerAlertMessage) { await next() }
}

export async function openSpamSlayerModal (suspiciousIpAddress, body, client) {
  const usersEmailAddress = await bolt.getUsersEmail(client, body.user.id)

  body.user.email = usersEmailAddress

  const modal = generateModal(suspiciousIpAddress, body)
  await client.views.open({
    trigger_id: body.trigger_id,
    view: modal
  })
}

export async function sendDemonetizedReport (viewMetadata) {
  const suspiciousIpAddress = viewMetadata.suspiciousIpAddress
  const startDate = viewMetadata.startDate
  const thread = viewMetadata.thread

  updateBtnViewBlockIp(viewMetadata).then(() => {
    api.getClicksReportSummaryFromTraction(suspiciousIpAddress, startDate).then((data) => {
      const filePath = tempWrite.sync(data, 'clicks-report.csv')
      uploadClickReportToSlack(filePath, suspiciousIpAddress, thread, true)
    })
  })
}

async function sendSupiciousIpAlertSlackResponse (suspiciousIpAddress, say) {
  const startOfMonth = util.getCurrentMonthStartDate()
  const today = util.getTodaysDate()

  const suspiciousIpAlertResponse = await bolt.generateSuspiciousIpAlertResponse(suspiciousIpAddress)

  const response = await say(suspiciousIpAlertResponse)
  const thread = response.ts

  try {
    const clickReportData = await api.getClicksReportFromTraction(suspiciousIpAddress, say)
    const reportFilePath = await tempWrite.sync(clickReportData, 'clicks-report.csv')
    await uploadClickReportToSlack(reportFilePath, suspiciousIpAddress, thread)
  } catch (error) {
    console.log(error)
  }

  try {
    const clicksReportSummary = await api.getClicksReportSummaryFromTraction(suspiciousIpAddress, say, startOfMonth, today)
    const summaryFilePath = tempWrite.sync(clicksReportSummary, `clicks-report-summary-${suspiciousIpAddress}.csv`)
    await uploadClickReportToSlack(summaryFilePath, suspiciousIpAddress, thread)
  } catch (error) {
    console.log(error)
  }
}

export async function updateBtnViewBlockIp (metadata) {
  let message = ''

  if (metadata.blockIp) {
    message += ` - block the IP Address: *${metadata.suspiciousIpAddress}* was successful\n`
  } else if (metadata.apiErrors.traffic) {
    message += ` -  block the IP Address: *${metadata.suspiciousIpAddress}* was *not* successful: ${metadata.apiErrors.traffic} \n`
  }

  message += ` - clear revenue starting from *${metadata.startDate}* was successful\nUpdated summary report will appear in this thread`

  const response = axios.post(metadata.responseUrl, {
    replace_original: true,
    text: `<@${metadata.user}> The process to:\n\n` + message
  }, (error) => {
    console.log(error)
  })
  return response
}

export async function updateBtnViewNoAction (ipAddress, user) {
  return `<@${user}> has chosen not to take action for ip address *${ipAddress}*`
}

export async function uploadClickReportToSlack (filepath, suspiciousIpAddress, thread, updated = false) {
  try {
    await bolt.app.client.files.uploadV2({
      channel_id: constant.hackathonChannel,
      thread_ts: thread,
      initial_comment: `${updated ? 'Updated' : ''} Click Revenue Report For IP: ${suspiciousIpAddress}`,
      file: fs.createReadStream(filepath),
      filename: `${suspiciousIpAddress}-revenue-report.csv`
    })
  } catch (error) {
    console.log(error)
  }
}

export function validateStartDate (date) {
  // Validate inputs
  const today = util.getTodaysDate()

  const errors = {}
  if (date > today) {
    errors['datepicker-block'] = 'Date Can\'t Be After Today!'
  }

  return errors
}
